/*
* Sketch: BCP_Example04_Servo_Advanced
* By Martyn Currey
* 17.09.2016
* Written in Arduino IDE 1.6.3
*
* Control a servo from a remote app
*
* Requires the  Bluetooth Control Panel Android App. Can be downloaded from Google Play.
* See http://www.martyncurrey.com/bluetooth-control-panel/

* Uses the following pins
* D5 - servo
* D8 - AltsoftSerial RX
* D9 - AltsoftSerial TX
*
*/

#include <AltSoftSerial.h>
AltSoftSerial BTserial; 


// ServoTimer2 library can be downloaded from https://github.com/nabontra/ServoTimer2 
// ServoTimer2 has a fairly high minimum pulse width and a fairly low maximum pilse width. These don't usually give full 180 degree movement.
// In the ServoTimer2.h file change the minimum and maximum pulse values:
// Change #define MIN_PULSE_WIDTH	750 to 500 
// Change #define MAX_PULSE_WIDTH	2250 to 2300

#include <ServoTimer2.h>  // the servo library
ServoTimer2 myservo;      // create a servo object

// The servo used in the example is at 0 degree when the pulse is 500 and at 180 degree (more like 178 degrees) when the pulse is 2300.
const int servoMaxPulse = 2250;
const int servoMinPulse = 510;

const boolean DEBUG = false;
 
// Variables used for incoming data
const byte maxDataLength = 20;
char receivedChars[21] ;
boolean newData = false;
boolean initialized = false;


// Vaiables 
unsigned long startTime = 0;
unsigned long flashStartTime = millis();
boolean LEDstate = LOW;

void setup()  
{
    // Set the onboard LED pin for output
    pinMode(13, OUTPUT); 
    digitalWrite(13,LOW);
    myservo.attach(3);
    myservo.write(1100); 
    
     
    if (DEBUG > 0)
    {
        // open serial communication for debugging
        Serial.begin(57600);
        Serial.println(__FILE__);
        Serial.println(__DATE__);
        Serial.println(" ");
    }
     
    //  open software serial connection to the Bluetooth module.
    BTserial.begin(9600); 
    if (DEBUG > 0)  {   Serial.println(F("BTserial started at 9600"));     }
          
    newData = false;
    initialized = false;
    startTime = millis();

  
} // void setup()
 
 

 
void loop()  
{
    if (!initialized)
    {
        waitForConnection();
    }
    
    if (initialized)
    {
       recvWithStartEndMarkers();                // check to see if we have received any new commands
       if (newData)  {   processCommand();  }    // if we have a new command do something about it
    }
}





/*
****************************************
* Function waitForConnection
* flashes the on board LED. Checks for incoming data
* checks for a CONNECT or RESET command
* 
* passed:
*  
* global: 
*       startTime
*       LEDstate
*       newData
*
* Returns:
*          
* Sets:
*       startTime
*       LEDstate
*       newData
*
*/
void waitForConnection()
{
      if (  millis()-startTime > 250 ) 
      {
           startTime = millis();
           if (LEDstate == HIGH) { LEDstate = LOW;  digitalWrite(13,LOW);  }
           else                  { LEDstate = HIGH; digitalWrite(13,HIGH); }
      }
       recvWithStartEndMarkers();                // check to see if we have received any new commands
       if (newData)  
       {   
            newData = false;
            if (strcmp ("CONNECT",receivedChars) == 0 or strcmp ("RESET",receivedChars) == 0) 
            { 
                 if (DEBUG) { Serial.println(F("CONNECT")); }
                 sendInitCommands();
                 
            }
       }  
  
}



/*
****************************************
* Function sendInitCommands
* Sends ascii initialization codes over BTserial
* 
* passed:
*  
* global: 
*       initialized
*
* Returns:
*          
* Sets:
*       initialized
*
*/
void sendInitCommands()
{ 
     BTserial.print("<T,Servo Control>");
     BTserial.print("<I,SS,1,Left,255250230>");
     BTserial.print("<I,SS,2,Centre,255250230>");
     BTserial.print("<I,SS,3,Right,255250230>");
     BTserial.print("<I,SL,1,Servo Control,0,180,255250230>");  
     BTserial.print("<I,DI,1,Servo Position,0,180,255255230>");        
     BTserial.print("<R,3>");  
     BTserial.print("<EOI>");  
     delay(100);
     initialized = true;
     digitalWrite(13,HIGH);     
     if (DEBUG) { Serial.println("Init commands sent"); }
}


       

 

/*
****************************************
* Function processCommand
* parses data commands contained in receivedChars[]
* receivedChars[] has not been checked for errors
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*
*/
void processCommand()
{
     newData = false;
     if (DEBUG) { Serial.println(""); Serial.print(F("receivedChars = "));  Serial.println(receivedChars);}

     if (strcmp ("RESET",receivedChars) == 0) 
     { 
          if (DEBUG) { Serial.println(F("RESET")); }
          sendInitCommands();
     }
     
     else if (strcmp ("DISCONNECT",receivedChars) == 0) 
     { 
          if (DEBUG) { Serial.println(F("DISCONNECT")); }
          initialized = false;
     }

    // Slider command = SL11234    
    else if (receivedChars[0]== 'S' && receivedChars[1]== 'L')
    {
          unsigned int servoPulse = 0;
          int val =   (receivedChars[3]-48) *1000;
          val = val + (receivedChars[4]-48) * 100;
          val = val + (receivedChars[5]-48) *  10;
          val = val + (receivedChars[6]-48);    
          
          //val = 180-val;
    
          if ( receivedChars[2]== '1' )
          {
              // convert the slider range 0-180 to the servo pulse range of 500-2400
              servoPulse = map(val,180,0,servoMinPulse,servoMaxPulse);
              
              if (DEBUG) { Serial.print("val = "); Serial.print(val);Serial.print("    servoPulse = "); Serial.println(servoPulse); }
              myservo.write(servoPulse);          
              BTserial.print("<D,DI,1,"); BTserial.print(val);  BTserial.print(">");          
           }
    }

    else if (receivedChars[0]== 'S' && receivedChars[1]== 'S')
    {
        int val = 0;
        unsigned int servoPulse = 0;
        if (receivedChars[2]== '1' && receivedChars[3]== '1')    // Left
        {
            val = 180;
            BTserial.print("<D,SS,1,0>");
        }
        else if (receivedChars[2]== '2' && receivedChars[3]== '1')    // Centre
        {
            val = 90;
            BTserial.print("<D,SS,2,0>");
        }
        else if (receivedChars[2]== '3' && receivedChars[3]== '1')    // Right
        {
            val = 0;
            BTserial.print("<D,SS,3,0>");
        }
        servoPulse = map(val,0,180,servoMinPulse,servoMaxPulse);
        myservo.write(servoPulse);  
        
        BTserial.print("<D,DI,1,"); BTserial.print(180-val);  BTserial.print(">");  
        BTserial.print("<D,SL,1,"); BTserial.print(180-val);  BTserial.print(">");  
        if (DEBUG) { Serial.print("val = "); Serial.print(180-val);Serial.print("    servoPulse = "); Serial.println(servoPulse); }
    }

    receivedChars[0] = '\0';
    strcpy("",receivedChars ); 
   
}





// function recvWithStartEndMarkers by Robin2 of the Arduino forums
// See  http://forum.arduino.cc/index.php?topic=288234.0
/*
****************************************
* Function recvWithStartEndMarkers
* reads serial data and returns the content between a start marker and an end marker.
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*       newData
*       receivedChars
*
*/
void recvWithStartEndMarkers()
{
     static boolean recvInProgress = false;
     static byte ndx = 0;
     char startMarker = '<';
     char endMarker = '>';
     char rc;
 
     if (BTserial.available() > 0) 
     {
          rc = BTserial.read();
          if (recvInProgress == true) 
          {
               if (rc != endMarker) 
               {
                    receivedChars[ndx] = rc;
                    ndx++;
                    if (ndx > maxDataLength) { ndx = maxDataLength; }
               }
               else 
               {
                     receivedChars[ndx] = '\0'; // terminate the string
                     recvInProgress = false;
                     ndx = 0;
                     newData = true;
               }
          }
          else if (rc == startMarker) { recvInProgress = true; }
     }
 
}







 
 
































