/*
* Sketch: BCP_Example_02_Large_Switch_LED_with_button_switch
* By Martyn Currey
* 20.07.2016
* Written in Arduino IDE 1.6.3
*
* Requires the Arduino Bluetooth Control II Android App. Can be downloaded from Google Play.
* See http://www.martyncurrey.com/arduino-bluetooth-control/
*
* Turn an LED on and off from an Android app or from a physical switch connected to the Arduino
* Uses the following pins
*
* D8 - AltsoftSerial TX
* D9 - AltsoftSerial RX
* D2 - LED
* D6 - Button switch
*
*/

// AltSoftSerial uses D9 for TX and D8 for RX. While using AltSoftSerial D10 cannot be used for PWM.
// Remember to use a voltage divider on the Arduino TX pin / Bluetooth RX pin
// Download from https://www.pjrc.com/teensy/td_libs_AltSoftSerial.html

#include <AltSoftSerial.h>
AltSoftSerial BTserial; 
 
// Change DEBUG to true to output debug information to the serial monitor
boolean DEBUG = true;

// Variables used for incoming data
const byte maxDataLength = 20;
char receivedChars[21] ;
boolean newData = false;
boolean initialized = false;

// Constants for hardware
const byte LED_RED_PIN = 2;
const byte SWITCH_PIN = 6;

// general variables
unsigned long flashStartTime = 0;
boolean LED_RED_State = false;
boolean switchState = false;
boolean oldSwitchState = false;


void setup()  
{
    // Set the onboard LED pin for output
    pinMode(13,OUTPUT); 
    digitalWrite(13,LOW);
     
    // Set the button switch pin for input
    pinMode(SWITCH_PIN, INPUT); 

    // Set the red LED pin for output
    pinMode(LED_RED_PIN, OUTPUT); 
    digitalWrite(LED_RED_PIN,LOW);
     
    if (DEBUG)
    {
        // open serial communication for debugging
        Serial.begin(9600);
        Serial.print("Sketch:   ");   Serial.println(__FILE__);
        Serial.print("Uploaded: ");   Serial.println(__DATE__);
        Serial.println(" ");
    }
     
    //  open software serial connection to the Bluetooth module.
    BTserial.begin(9600); 
    if (DEBUG)  { Serial.println("AltSoftSerial started at 9600"); }
          
    newData = false;
    initialized = false;
    
    flashStartTime = millis();

} // void setup()
 
 

 
void loop()  
{
    if (!initialized)
    {
        // Flash the built in LED to show the Arduino is waiting for the initialization commands
        if (  millis()-flashStartTime > 250 ) 
        {
             flashStartTime = millis();
             if (digitalRead(13) == HIGH) { digitalWrite(13,LOW);  }
             else                         { digitalWrite(13,HIGH); }
        }
         recvWithStartEndMarkers();                // check to see if we have received any new commands
         if (newData)  {   processCommand();  }    // if we have a new command do something
    }

    // after the initialization commands are received keep checking the button switch and checking for in coming data    
    if (initialized)
    {
       checkSwitch();
       recvWithStartEndMarkers();                // check to see if we have received any new commands
       if (newData)  {   processCommand();  }    // if we have a new command do something about it
    }
}




 


/*
****************************************
* Function checkSwitch()
* checks the status of a button switch and turns an LED on or off depending on the switch status
* 
* passed:
*  
* global: 
*      switchState
*      LED_RED_State
*      oldSwitchState
*
* Returns:
*          
* Sets:
*      switchState
*      LED_RED_State
*      oldSwitchState
*/
void checkSwitch()
{
     // Simple toggle switch function with very simple debouce.
     boolean state1 = digitalRead(SWITCH_PIN);
     boolean state2 = digitalRead(SWITCH_PIN);
     boolean state3 = digitalRead(SWITCH_PIN);
     if ((state1 == state2) && (state1==state3))   
     { 
          switchState = state1;  
          if ( (switchState == HIGH) && (oldSwitchState == LOW) )
          {
               LED_RED_State = ! LED_RED_State;  
               if ( LED_RED_State == HIGH) {  
                                               BTserial.print("<D,SW,1,1>" );  
                                               digitalWrite(LED_RED_PIN,HIGH);  
                                               if (DEBUG) { Serial.println("Sent - <D,SW,1,1>");   } 
                                           }
                                           
               else                        {  
                                               BTserial.print("<D,SW,1,0>");   
                                               digitalWrite(LED_RED_PIN,LOW);   
                                               if (DEBUG) { Serial.println("Sent - <D,SW,1,0>");   } 
                                           }    
          }          
          oldSwitchState = switchState;
      }
}



/*
****************************************
* Function reset
* sets the LED pin LOW
* 
* passed:
*  
* global: 
*
* Returns:
*          
* Sets:
*/
void reset()
{
    digitalWrite(LED_RED_PIN,LOW);
}




/*
****************************************
* Function sendInitCommands
* sends the initialization commands by software serial
* 
* passed:
*  
* global: 
*       initialized
*
* Returns:
*          
* Sets:
*       initialized
*/
 void sendInitCommands()
{ 

     BTserial.print("<T,Red LED Control>");
     BTserial.print("<I,SW,1,Red LED,255235230>"); 
     BTserial.print("<R,5>");  
     BTserial.print("<EOI>");  
     delay(100);
     
     initialized = true;
     digitalWrite(13,HIGH);
     
     if (DEBUG) { Serial.println("Init commands sent"); }     
     
}



/*
****************************************
* Function processCommand
* parses data commands contained in receivedChars[]
* receivedChars[] has not been checked for errors
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*       receivedChars[]
*       newData
*/
void processCommand()
{
    if (DEBUG)
    {
         Serial.print("receivedChars = ");  
         Serial.println(receivedChars);
    }


     if (strcmp ("CONNECT",receivedChars) == 0) 
     { 
          Serial.println("CONNECT message received"); 
          sendInitCommands();
     }

     else if (strcmp ("RESET",receivedChars) == 0) 
     { 
          if (DEBUG) { Serial.println("RESET message received"); }
          reset();
          sendInitCommands();
     }
     
     else if (strcmp ("DISCONNECT",receivedChars) == 0) 
     { 
          if (DEBUG) { Serial.println("DISCONNECT message received"); }
          reset();
          initialized = false;
     }

    if (strcmp ("SW10",receivedChars) == 0) 
    {
        digitalWrite(LED_RED_PIN,LOW);
        LED_RED_State = LOW; 
        if (DEBUG) { Serial.println("LED LOW");  }
    }
    
    else if (strcmp ("SW11",receivedChars) == 0)
    {
        digitalWrite(LED_RED_PIN,HIGH);
        LED_RED_State = HIGH; 
        if (DEBUG) { Serial.println("LED HIGH");  }
    }
   
    receivedChars[0] = '\0';
    newData = false;
   
}


// function recvWithStartEndMarkers by Robin2 of the Arduino forums
// See  http://forum.arduino.cc/index.php?topic=288234.0
/*
****************************************
* Function recvWithStartEndMarkers
* reads serial data and returns the content between a start marker and an end marker.
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*       newData
*       receivedChars
*
*/
void recvWithStartEndMarkers()
{
     static boolean recvInProgress = false;
     static byte ndx = 0;
     char startMarker = '<';
     char endMarker = '>';
     char rc;
 
     if (BTserial.available() > 0) 
     {
          rc = BTserial.read();
          if (recvInProgress == true) 
          {
               if (rc != endMarker) 
               {
                    receivedChars[ndx] = rc;
                    ndx++;
                    if (ndx > maxDataLength) { ndx = maxDataLength; }
               }
               else 
               {
                     receivedChars[ndx] = '\0'; // terminate the string
                     recvInProgress = false;
                     ndx = 0;
                     newData = true;
               }
          }
          else if (rc == startMarker) { recvInProgress = true; }
     }
}



 
 
































