/*
* Sketch: ArduinoSketch_ArduinoBluetoothControl_RESET
* By Martyn Currey
* 13.02.2016
* Written in Arduino IDE 1.6.3
*
* Requires the Arduino Bluetooth Control Android App.
*
* See http://www.martyncurrey.com/arduino-bluetooth-control/
*
* Read pin states / values and sends to an Android app over bluetooth
* Receive control commands from the Android app
* The Android app is initialized by commands sent from the Arduino
*
* The example uses the following pins
* A1 - a potentiometer
* D2 - software serial RX
* D3 - software serial TX
*/
 
// The Bluetooth module is connected to pin D2 and D3 and uses software serial
#include <SoftwareSerial.h>
SoftwareSerial BTserial(2,3); // RX | TX
 
// Change DEBUG to true to output debug information to the serial monitor
const boolean DEBUG = true;
 
char numberString[10];
// used to hold an ascii representation of a number
// [10] allows for 9 digits but in this example I am only using 4 digits
 
char rc = ' ';
const byte maxDataLength = 20;
char receivedChars[21] ;
boolean newData = false;
 
unsigned long startTime = 0;
unsigned long refreshRate = 100;
 
// I only send a value if the value of the sensor has changed.
// This means I need to store the old value so I can compare.
unsigned int oldPotVal = 0;
unsigned int newPotVal = 0;

boolean initialized = false;


 
void setup()  
{
 
  // potPin (A1) does not need initializing
 
  //Pin 13 LED is used to show status 
  // Flashing means sending initialization commands and waiting for a reply
  // On means the Android app has replied correctly
  pinMode(13, OUTPUT); 
  digitalWrite(13,LOW);
 
  if (DEBUG)
  {
      // open serial communication for debugging
      Serial.begin(9600);
      Serial.println(__FILE__);
      Serial.println(__DATE__);
      Serial.println(" ");
  }

  BTserial.begin(9600);   //  open software serial connection to the Bluetooth module
  refreshRate = 100;      // refeshRate is the frequency to check the input pins
 
} // void setup()
 
 


void loop()  
{

     if (!initialized)
     {
         sendInitCommands();  // send the init commands and wait for a reply.
       
     }
     else
     {
        if ( millis()-refreshRate > startTime)
        {
          startTime = millis();
          checkPins();  // read the pin values and if the values have changed send the new values to the Android app
        }
       
        recvWithStartEndMarkers();          // check to see if we have received any new commands
        if (newData) { processCommand(); }  // if we have a new command do something about it
       
    }

}
 
 


/*
****************************************
* Function sendInitCommands
* Sends initialization commands and then waits for a "OK" reply.
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*
*/
void sendInitCommands()
{
  // Send initialization commands and the wait for an "OK"   
  boolean LEDisON = false;
  boolean done = false;
  byte count = 0;
  boolean sendInitCodes = true;
 
  newData = false;
  long unsigned startTime = millis();
  while (!done)
  {
        // This bit flashes the LED on pin 13. 
        if ( millis()-startTime > 500 )
        {
            startTime = millis();
 
            if (LEDisON) { LEDisON = false;  digitalWrite(13,LOW);   }
            else         { LEDisON = true;   digitalWrite(13,HIGH);  } 
 
            count++; 
            // if reply is not received within 3 seconds resend the commands
            if (count>=6) { count = 0; sendInitCodes = true;         }
        }
 
 
        // Keep sending the initialization commands until the "OK" reply is received.
        if (sendInitCodes == true)   
        { 
            
            dumpBTserialBuffer(); // not really required but it makes me feel better :-)
            BTserial.print("<START>");                // Start marker
            BTserial.print("<A010000>");              // Initialize analogue pin A01. Start value 0000
            BTserial.print("<NA01Potentiometer>");    // set the pin description for A1 to Potentiometer. This will be truncated      
            BTserial.print("<C2>");                   // number of commands not including itself
            BTserial.print("<END>");                  // End marker
            sendInitCodes = false; 
            
            if (DEBUG) { Serial.println("Sent init commands. Waiting for reply"); }
        }
 
 
        recvWithStartEndMarkers(); // check for new data from the Bluetooth module
        if (newData)
        {
            // The Android app receives the commands and sends an "OK" back.
            if (strcmp ("OK",receivedChars) == 0)   { done = true;  }
            else                                    { newData = false; }            
 
            if (DEBUG) { Serial.print("receivedChars = "); Serial.println( receivedChars );  }
        }
 
  } // while (!done)   

  initialized = true;
  
  // Turn on the built in LED to show the app has received the initialization codes
  digitalWrite(13,HIGH); 
  if (DEBUG)  {  Serial.println("OK received from the app");  } 

  
}


 
/*
****************************************
* dumpBTserialBuffer
* removes data from the software serial buffer
* 
* passed:
*  
* global: 
*
* Returns:
*          
* Sets:
*
*/
void dumpBTserialBuffer()
{
     while(BTserial.available() > 0)  { char t = BTserial.read();  }
}  


 
 
/*
****************************************
* Function processCommand
* parses data commands contained in receivedChars[]
* receivedChars[] has not been checked for errors
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*
*/
void processCommand()
{
     newData = false;
     
     if (strcmp ("RESET",receivedChars) == 0) 
     {
         initialized = false;
     }
     
     if (strcmp ("DISCONECT",receivedChars) == 0) 
     {
         initialized = false;
     }
     
     if (DEBUG) { Serial.print("Recieved data = ");  Serial.println(receivedChars); }
 
}
 


 
/*
****************************************
* Function checkPins
* Checks the values of the deignated pins and if the value has changed sends ascii codes to software serial
* 
* passed:
*  
* global: 
*         newPotVal
*         oldPotVal
*         potPin
*         newButtonSwitchState
*         oldButtonSwitchState
*         buttonSwitchPin
*         numberString
*
* Returns:
*          
* Sets:
*
*/
void checkPins()
{
        // Check the value on analogue pin A1
        newPotVal = analogRead(A1);
        //The pot I am using jitters +/- 1 so I only using changes of 2 or more.  
        if ( abs(newPotVal-oldPotVal) > 1)
        {
             oldPotVal = newPotVal;
             formatNumber( newPotVal, 4);
             BTserial.print("[A01");
             BTserial.print(numberString);
             BTserial.print("]");      
 
             if (DEBUG) { Serial.print("[A01"); Serial.print(numberString); Serial.println("]"); }
        }   


}
 
 
/*
****************************************
* Function formatNumber
* formats a number in to a string and copies it to the global char array numberString
* pads the start of the string with '0' characters
* 
* passed:
*         number = the integer to convert to a string
*         digits = the number of digits to use         
*  
* global: 
*         numberString
*
* Returns:
*          
* Sets:
*         numberString
*
*/
void formatNumber( unsigned int number, byte digits)
{
    char tempString[10] = "\0"; 
    strcpy(numberString, tempString);
 
    // convert an integer into a acsii string base 10
    itoa (number, tempString, 10);
 
    // create a string of '0' characters to pad the number    
    byte numZeros = digits - strlen(tempString) ;
    if (numZeros > 0)
    {
       for (int i=1; i <= numZeros; i++)    { strcat(numberString,"0");  }
    }
    strcat(numberString,tempString); 
}
 
 
// function recvWithStartEndMarkers by Robin2 of the Arduino forums
// See  http://forum.arduino.cc/index.php?topic=288234.0
/*
****************************************
* Function recvWithStartEndMarkers
* reads serial data and returns the content between a start marker and an end marker.
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*       newData
*       receivedChars
*
*/
void recvWithStartEndMarkers() 
{
     static boolean recvInProgress = false;
     static byte ndx = 0;
     char startMarker = '<';
     char endMarker = '>';
 
     if (BTserial.available() > 0) 
     {
          rc = BTserial.read();
 
          if (recvInProgress == true) 
          {
               if (rc != endMarker) 
               {
                    if (ndx < maxDataLength)
                    {
                       receivedChars[ndx] = rc;
                       ndx++;
                    }
               }
               else 
               {
                     receivedChars[ndx] = '\0'; // terminate the string
                     recvInProgress = false;
                     ndx = 0;
                     newData = true;
               }
          }
          else if (rc == startMarker) { recvInProgress = true; }
     }
}
