/*
* Sketch: ArduinoSketch_ArduinoBluetoothControl_example003b
* By Martyn Currey
* 01.07.2015
* Written in Arduino IDE 1.6.3
*
* Requires the Arduino Bluetooth Control Android App.
*
* See http://www.martyncurrey.com/arduino-bluetooth-control/
*
* Read pin states / values and send to an Android app over bluetooth
* Receive control commands from the Android app
* The Android app is initialized by commands sent from the Arduino
*
* The example uses the following pins
* A1 - a potentiometer
* D2 - software serial RX
* D3 - software serial TX
* D6 - button switch (pulled LOW)
* D10 - LED
* D11 - LED
* D12 - LED
*/
 
// The Bluetooth module is connected to pin D2 and D3 and uses software serial
#include <SoftwareSerial.h>
SoftwareSerial BTserial(2,3); // RX | TX
 
// Change DEBUG to true to output debug information to the serial monitor
const boolean DEBUG = true;
 
char numberString[10];
// used to hold an ascii representation of a number
// [10] allows for 9 digits but in this example I am only using 4 digits
 
char rc = ' ';
const byte maxDataLength = 20;
char receivedChars[21] ;
boolean newData = false;
 
unsigned long startTime = 0;
unsigned long refreshRate = 100;
 
// I only send a value if the value has changed.
// This means I need to store the old value
unsigned int oldPotVal = 0;
unsigned int newPotVal = 0;
boolean oldButtonSwitchState = false;
boolean newButtonSwitchState = false;
 
 
void setup()  
{
  // set the button switch pin to input
  pinMode(6, INPUT); 
 
  // Set the LED pins to output and turn them off.
  pinMode(12, OUTPUT); // Pin 12
  pinMode(11, OUTPUT); // Pin 11 
  pinMode(10, OUTPUT); // Pin 10
  digitalWrite(12,LOW);
  digitalWrite(11,LOW);  
  digitalWrite(10,LOW);
 
  // potPin (A1) does not need initializing
 
  //Pin 13 LED is used to show status 
  // Flashing means sending initialization commands and waiting for a reply
  // On means the Android app has replied correctly
  pinMode(13, OUTPUT); 
  digitalWrite(13,LOW);
 
 
  if (DEBUG)
  {
      // open serial communication for debugging
      Serial.begin(9600);
      Serial.println("ArduinoSketch_ArduinoBluetoothControl_example003b");
      Serial.println(" ");
  }
 
 
  //  open software serial connection to the Bluetooth module
  BTserial.begin(9600); 
 
  // Send initialization commands and the wait for an "OK"   
  boolean LEDisON = false;
  boolean done = false;
  byte count = 0;
  boolean sendInitCodes = true;
 
  newData = false;
  refreshRate = 500;
  startTime = millis();
  while (!done)
  {
        // This bit flashes the LED on pin 13. 
        if ( millis()-startTime > refreshRate)
        {
            startTime = millis();
 
            if (LEDisON) { LEDisON = false;  digitalWrite(13,LOW);   }
            else         { LEDisON = true;   digitalWrite(13,HIGH);  } 
 
            count++; 
            // if reply is not received within 3 seconds resend the commands
            if (count==6) { count = 0; sendInitCodes = true;         }
        }
 
 
        // Keep sending the initialization commands until the "OK" reply is received.
        if (sendInitCodes == true)   
        { 
            sendInitCodes = false; 
            dumpBTserialBuffer(); // not really required but it makes me feel better :-)
 
            BTserial.print("<START>");                // Start marker
            BTserial.print("<A010000>");              // Initialize analogue pin A01. Start value 0
            BTserial.print("<D06IL>");                // Set D6 to input. Start value = LOW
            BTserial.print("<P10000255000>");         // Initialize D10 for PWN. Start value = 0
            BTserial.print("<D11OL>");                // set D11 to output. Start value = LOW
            BTserial.print("<D12OL>");                // set D12 to output. Start value = LOW
            BTserial.print("<ND12LED #1>");           // set the pin description for D12 to LED #1
            BTserial.print("<ND11LED #2>");           // set the pin description for D11 to LED #2
            BTserial.print("<ND10LED Fader>");        // set the pin description for D10 to LED Fader
            BTserial.print("<ND06BTN Switch>");       // set the pin description for D6 to BTN Switch
            BTserial.print("<NA01Potentiometer>");    // set the pin description for A1 to Potentiometer. This will be truncated            
            BTserial.print("<C10>");                  // number of commands not including itself.
            BTserial.print("<END>");                  // End marker
 
            if (DEBUG) { Serial.println("Sent init commands. Waiting for receipt"); }
        }
 
 
        recvWithStartEndMarkers(); // check for new data from the Bluetooth module
        if (newData)
        {
            // The Android app receives the commands and sends an "OK" back.
            if (  (receivedChars[0] == 'O') & (receivedChars[1] == 'K')  )
            { done = true;  }
            else 
            {newData = false; }            
 
            if (DEBUG) { Serial.print("receivedChars = "); Serial.println( receivedChars );  }
        }
 
  } // while (!done) 
 
 
  // Turn on the built in LED to show the app has received the initialization codes
  digitalWrite(13,HIGH); 
  if (DEBUG)  {  Serial.println("OK received from the app");  } 
 
 
  // refeshRate is the frequency to check the input pins
  refreshRate = 100;
 
} // void setup()
 
 
void loop()  
{
 
  if ( millis()-refreshRate > startTime)
  {
    startTime = millis();
    checkPins();  // read the pin values and if the values have changed send the new values to the Android app
  }
 
  recvWithStartEndMarkers();          // check to see if we have received any new commands
  if (newData) { processCommand(); }  // if we have a new command set the Arduino pin accordingly
 
}
 
 
/*
****************************************
* dumpBTserialBuffer
* removes data from the software serial buffer
* 
* passed:
*  
* global: 
*
* Returns:
*          
* Sets:
*
*/
void dumpBTserialBuffer()
{
     while(BTserial.available() > 0)  { char t = BTserial.read();  }
}  
 
 
/*
****************************************
* Function processCommand
* parses data commands contained in receivedChars[]
* receivedChars[] has not been checked for errors
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*
*/
void processCommand()
{
     newData = false;
     byte pin = ((receivedChars[1]-48) *10 ) + receivedChars[2]-48;
 
     // digital pin 
     if ( receivedChars[0] == 'D')
     {
          if ( receivedChars[3] == 'H') { digitalWrite(pin,HIGH); }
          if ( receivedChars[3] == 'L') { digitalWrite(pin,LOW);  }
          if (DEBUG) { Serial.print("Digital pin command = ");  Serial.println(receivedChars); }
     }
 
     // PWM pin
     if ( receivedChars[0] == 'P')
     {
          byte val = ((receivedChars[3]-48) *100 ) + ((receivedChars[4]-48) * 10)  + (receivedChars[5]-48)   ;
          analogWrite(pin, val);
          if (DEBUG) { Serial.print("PWN command = ");  Serial.println(receivedChars); }
     }  
}
 
 
/*
****************************************
* Function checkPins
* Checks the values of the deignated pins and if the value has changed sends ascii codes to software serial
* 
* passed:
*  
* global: 
*         newPotVal
*         oldPotVal
*         potPin
*         newButtonSwitchState
*         oldButtonSwitchState
*         buttonSwitchPin
*         numberString
*
* Returns:
*          
* Sets:
*
*/
void checkPins()
{
        // Check the value on analogue pin A1
        newPotVal = analogRead(A1);
        //The pot I am using jitters +/- 1 so I only using changes of 2 or more.  
        if ( abs(newPotVal-oldPotVal) > 1)
        {
             oldPotVal = newPotVal;
             formatNumber( newPotVal, 4);
             BTserial.print("[A01");
             BTserial.print(numberString);
             BTserial.print("]");      
 
             if (DEBUG) { Serial.print("[A01"); Serial.print(numberString); Serial.println("]"); }
        }   
 
 
 
        // Check the state of pin 6 - the button switch.        
        newButtonSwitchState = digitalRead(6);
        if (newButtonSwitchState != oldButtonSwitchState)
        {
             oldButtonSwitchState = newButtonSwitchState;
             if (oldButtonSwitchState == true)
             {
                if (DEBUG) { Serial.println("[D06H]"); }
                BTserial.print("[D06H]");
             }
             else
             {
                if (DEBUG) { Serial.println("[D06L]"); }
                BTserial.print("[D06L]");
             }
        } 
}
 
 
/*
****************************************
* Function formatNumber
* formats a number in to a string and copies it to the global char array numberString
* pads the start of the string with '0' characters
* 
* passed:
*         number = the integer to convert to a string
*         digits = the number of digits to use         
*  
* global: 
*         numberString
*
* Returns:
*          
* Sets:
*         numberString
*
*/
void formatNumber( unsigned int number, byte digits)
{
    char tempString[10] = "\0"; 
    strcpy(numberString, tempString);
 
    // convert an integer into a acsii string base 10
    itoa (number, tempString, 10);
 
    // create a string of '0' characters to pad the number    
    byte numZeros = digits - strlen(tempString) ;
    if (numZeros > 0)
    {
       for (int i=1; i <= numZeros; i++)    { strcat(numberString,"0");  }
    }
    strcat(numberString,tempString); 
}
 
 
// function recvWithStartEndMarkers by Robin2 of the Arduino forums
// See  http://forum.arduino.cc/index.php?topic=288234.0
/*
****************************************
* Function recvWithStartEndMarkers
* reads serial data and returns the content between a start marker and an end marker.
* 
* passed:
*  
* global: 
*       receivedChars[]
*       newData
*
* Returns:
*          
* Sets:
*       newData
*       receivedChars
*
*/
void recvWithStartEndMarkers() 
{
     static boolean recvInProgress = false;
     static byte ndx = 0;
     char startMarker = '<';
     char endMarker = '>';
 
     if (BTserial.available() > 0) 
     {
          rc = BTserial.read();
 
          if (recvInProgress == true) 
          {
               if (rc != endMarker) 
               {
                    receivedChars[ndx] = rc;
                    ndx++;
                    if (ndx > maxDataLength) { ndx = maxDataLength; }
               }
               else 
               {
                     receivedChars[ndx] = '\0'; // terminate the string
                     recvInProgress = false;
                     ndx = 0;
                     newData = true;
               }
          }
          else if (rc == startMarker) { recvInProgress = true; }
     }
}
