/* 
* Sketch Arduino and Visual Basic Part 3 
* Send and receive between Arduino and Visual Basic
* https://www.martyncurrey.com/arduino-and-visual-basic-part-3-receiving-data-from-the-arduino/
* 
* potentiometer used to control LED PWM
* push button switch, turn LED on and off
*/

/*
* Pins
* D3 - LED + resistor
* D4 - push button switch
* D5 - control LED + resistor
* A4 - potentiometer
*/
 
// When DEGUG is TRUE print newline characters
// Useful when using the serial monitor 
const boolean DEBUG = false;
 
const byte redLedPin       = 3;
const byte buttonSwitchPin = 4;
const byte controlLEDPin   = 5;
const byte potPin          = A4;

boolean newSwitchState1 = LOW;
boolean newSwitchState2 = LOW;
boolean newSwitchState3 = LOW;
boolean oldSwitchState  = LOW;

unsigned int oldPotVal = 0;
unsigned int newPotVal = 0;
const int potUpdateFreq = 200;  //  how often the potentiometer is read
long timeNow  = 0;
long timePrev = 0;
 
// used to hold an ascii representation of a number [10] allows for 9 digits 
char numberString[10];

boolean haveNewData = false;
boolean ArduinoHasControl = false;  // false used for off

char recDataBuffer [31];
int length = 30;
 
 
void setup()  
{
  pinMode(redLedPin, OUTPUT);       digitalWrite(redLedPin ,LOW);
  pinMode(controlLEDPin , OUTPUT);  digitalWrite(controlLEDPin  ,LOW);
  
  pinMode(buttonSwitchPin, INPUT); 

  Serial.begin(115200);
  while (!Serial) {;}
  Serial.println("Adruino is ready");
  Serial.println(" ");

  Serial.print("<ARD_ON>"); 
  if (DEBUG) { Serial.println(""); }    
  turnArduinoControlOn();
}
 
 
void loop()  
{
  checkSerialIn();
  if (haveNewData)   { processNewData(); }
    
  checkPushButtonSwitch();
    
  if (ArduinoHasControl)
  {  
      timeNow = millis();
      if (timeNow - timePrev >= potUpdateFreq )    
      {   
            timePrev = timeNow;   
            updatePotentiometer();
      }
  } 

}  //  loop()
 
  
	
void turnArduinoControlOn()
{
  ArduinoHasControl = true; 
  oldPotVal =0;
  updatePotentiometer();
  digitalWrite(controlLEDPin ,HIGH);
}

void turnArduinoControlOff()
{
  ArduinoHasControl = false; 
  digitalWrite(controlLEDPin,LOW);
}

	

void checkSerialIn()
{
		// checks serial in.
		// if data is available, and if the start marker is received, copies it to a buffer
		// when the end marker is received stop copying and set haveNewData = true
		static boolean recvInProgress = false;
		static byte ndx = 0;
		char startMarker = '<';
		char endMarker = '>';
		char rc;

		if (Serial.available() > 0) 
		{
				rc = Serial.read();

				if (recvInProgress == true) 
				{
						 if (rc != endMarker) 
						 {
									recDataBuffer[ndx] = rc;
									ndx++;
									if (ndx > length) { ndx = length; }
						 }
						 else 
						 {
									 recDataBuffer[ndx] = '\0'; // terminate the string
									 recvInProgress = false;
									 ndx = 0;
									 haveNewData = true;
						 }
				}
				else if (rc == startMarker) { recvInProgress = true; }
		}
}


void processNewData()
{
  
  if      (strcmp(recDataBuffer, "VB_ON")  == 0)  { turnArduinoControlOff();  haveNewData = false; }
  else if (strcmp(recDataBuffer, "VB_OFF") == 0)  { turnArduinoControlOn();   haveNewData = false; }

  // LB = LED Brightness
  else if (recDataBuffer[0] == 'L' && recDataBuffer[1] == 'B' )   
  {
      // convert the ascii numbers to an actual number
      int LED_PWN_val = 0;
      LED_PWN_val  =              (recDataBuffer[2] - 48) * 100;
      LED_PWN_val = LED_PWN_val + (recDataBuffer[3] - 48) * 10;
      LED_PWN_val = LED_PWN_val +  recDataBuffer[4] - 48;

      analogWrite(redLedPin,LED_PWN_val);
      haveNewData = false; 
  }
	
	else
	{
			// unknown command  
			// issue an error message in some way or simply set haveNewData to false and ignore
			haveNewData = false;  
	}

}



void checkPushButtonSwitch()
{
    // check the push button switch pin and if pressed toggle Arduino control / App control
    
    // simple debounce
    newSwitchState1 = digitalRead(buttonSwitchPin);      delay(1);
    newSwitchState2 = digitalRead(buttonSwitchPin);      delay(1);
    newSwitchState3 = digitalRead(buttonSwitchPin);
 
    if (  (newSwitchState1==newSwitchState2) && (newSwitchState1==newSwitchState3) )
    {
				if ( newSwitchState1 != oldSwitchState ) 
				{
						oldSwitchState = newSwitchState1;

					 // has the button switch been closed?
					 if ( newSwitchState1 == HIGH )
					 {
								if ( ArduinoHasControl == false ) 
								{ 
										Serial.print("<ARD_ON>"); 
										if (DEBUG) { Serial.println(""); }                      
										turnArduinoControlOn();
								} 
								else                              
								{ 
									Serial.print("<ARD_OFF>");  
									if (DEBUG) { Serial.println(""); }  							
									turnArduinoControlOff();
								}
						}  
       } 
    }

}  //  checkPushButtonSwitch()


void updatePotentiometer()
{
    // get the potentiometer value and set the LED PWN accordingly
    // pot value = 0 to 1023, LED PWN is 0 to 255, so we divide the pot value by 4
    // LED brightness is logarithmic not linera but I don't worry about it here.
    
    // The potentiometer I am using jitters a bit (cheap breadboard I think) so I only using changes of 5 or more.  
    newPotVal = analogRead(potPin); 
    if ( abs(newPotVal-oldPotVal) > 4)
    {
        oldPotVal = newPotVal;  
        
        int LED_PWN_val = newPotVal / 4;
        analogWrite(redLedPin,LED_PWN_val);
        
        formatNumber( LED_PWN_val, 3);
        Serial.print("<LB");   
        Serial.print(numberString);     
        Serial.print(">");
        if (DEBUG) { Serial.println("");  }   
    }   
 
}  // updatePotentiometer()



void formatNumber( unsigned int number, byte digits)
{
    // Formats a number in to a string and copies it to the global char array numberString
    // pads the start of the string with '0' characters
    //
    // number = the integer to convert to a string
    // digits = the number of digits to use. 
 
    char tempString[10] = "\0"; 
    strcpy(numberString, tempString);
 
    // convert an integer into a acsii string
    itoa (number, tempString, 10);
 
    // create a string of '0' characters to pad the number    
    byte numZeros = digits - strlen(tempString) ;
    if (numZeros > 0)
    {
       for (int i=1; i <= numZeros; i++)    { strcat(numberString,"0");  }
    }
     strcat(numberString,tempString); 
} // formatNumber
